#include "formats.h"

/* This converts the s bits in src into at most d chars in dest using a
   generic format described by the number of bits per character, the
   start sentinel, the end sentinel, and the ASCII offset. Returns the
   number of characters stored in dest, -1 if src contains a bad
   checksum, -2 if src contains a bad LRC. */
long generic_format(char *dest, const unsigned char *src, const long s,
    const long d, const int use_checksums, const int bits_per_char,
    const int start_sen, const int end_sen, const int ascii_offset) {

  unsigned char *tmp_uchptr = (unsigned char *) src;
  unsigned char tmp_uchar, odd, lrc = 0;
  long check = 0, written = 0;
  int i;

  /* Find start sentinel */
  while (check < s - (bits_per_char - 1)) {
    tmp_uchar = 0;
    for (i = 0; i < bits_per_char - 1; i++)
      tmp_uchar |= ((tmp_uchptr[i] & 0x01) << i);

    if (tmp_uchar == start_sen) {
      /* Calculate checksum and LRC */
      odd = 0;
      lrc = 0;
      for (i = 0; i < bits_per_char; i++) {
        odd ^= (tmp_uchptr[i] & 0x01);
        lrc ^= (tmp_uchptr[i] & 0x01);
      }

      /* Start sentinel found, if checksum is odd */
      if (!use_checksums || odd)
        break;
    }

    check++;
    tmp_uchptr++;
  }

  // Parsing actual data until end sentinel or end of dest */
  tmp_uchptr += bits_per_char;
  check += bits_per_char;
  while (check < s - (bits_per_char - 1) && written < d) {
    tmp_uchar = 0;
    for (i = 0; i < bits_per_char - 1; i++)
      tmp_uchar |= ((tmp_uchptr[i] & 0x01) << i);

    /* Calculate checksum and LRC */
    odd = 0;
    for (i = 0; i < bits_per_char; i++) {
      odd ^= (tmp_uchptr[i] & 0x01);
      lrc ^= (tmp_uchptr[i] & 0x01);
    }

    /* Bad checksum */
    if (use_checksums && !odd)
      return -1;

    /* End sentinel found */
    if (tmp_uchar == end_sen)
      break;

    dest[written] = tmp_uchar + ascii_offset;

    check += bits_per_char;
    tmp_uchptr += bits_per_char;
    written++;
  }

  /* Read LRC */
  tmp_uchptr += bits_per_char;
  check += bits_per_char;
  if (check < s - (bits_per_char - 1)) {
    /* Calculate LRC */
    for (i = 0; i < bits_per_char - 1; i++)
      lrc |= (tmp_uchptr[i] & 0x01);

    /* Bad LRC */
    if (use_checksums && !lrc)
      return -2;
  }

  return written;
}

/* This converts the s bits in src into at most d chars in dest using
   the ANSI/ISO ALPHA data format. Returns number of characters stored
   in dest, -1 if src contains a bad checksum, -2 if src contains a bad
   LRC. */
long ansi_iso_alpha(char *dest, const unsigned char *src, const long s,
    const long d, const int use_checksums) {
  return generic_format(dest, src, s, d, use_checksums, 7, 0x05, 0x1F,
    0x20);
}

/* This converts the s bits in src into at most d chars in dest using
   the ANSI/ISO BCD data format. Returns number of characters stored in
   dest, -1 if src contains a bad checksum, -2 if src contains a bad
   LRC. */
long ansi_iso_bcd(char *dest, const unsigned char *src, const long s,
    const long d, const int use_checksums) {
  return generic_format(dest, src, s, d, use_checksums, 5, 0x0B, 0x0F,
    0x30);
}
